<?PHP
if (! defined ( 'BASEPATH' )) exit ( 'No direct script access allowed' );

/**
 * @package direct-as-a-service
 * @subpackage models
 * @filesource
 */

/** */
require_once 'entity.php';
require_once APPPATH.'libraries/ccd.php';

/**
 * @package direct-as-a-service
 * @subpackage models           
 */
class Attachment_disclosure extends Entity {
	static $database_group = 'mail_db';
	static $table = 'accounting_disclosure';	
	
	protected $_attachment_content;
	protected $_attachment_parser;
	
	protected static $_relationships = array( 'message' => array('type' => 'belongs_to', 'related_foreign_key' => 'messages_table_id') );	
	
	protected $_readonly_fields = array( 'message_id', 'first', 'last', 'title', 'received_from', 'username', 'patient_id');
	
	protected $_property_validation_rules = array('recipient' => 'string_like_an_email_address');
	
	
	public function has_message(){
		$message_foreign_key = static::related_foreign_key('message');
		return isset($message_foreign_key) && Message::formatted_like_an_id($this->$message_foreign_key);
	}
	
#TODO - set hash -> need to change the values
#TODO - set message_table_id -> need to change the values

	public function set_hash($value, $offset=0){
		if(!is_string($value) || strlen($value) != 40) 
			return $this->error->should_be_an_x( $value, 'valid value ('.array_to_human_readable_list($valid_values, 'or').')', $offset+1);
			
		//wipe the content of the attachment content and the attachment parser, force a reload	
		if(isset($this->_attachment_content)) $this->_attachment_content = null;
		if(isset($this->_attachment_parser)) $this->_attachment_parser = null;
		
		$this->_set_field_value('hash', $value, $offset+1);
	}

	
	/////////////////////////////
	// GETTERS
	/////////////////////////////
	
	public function first(){
		if(is_object($this->attachment_parser)) return implode_nonempty(' ', $this->attachment_parser->given_names);
	}
	
	public function last(){
		if(is_object($this->attachment_parser)) return $this->attachment_parser->family_name;
	}
	
	public function title(){
		if(is_object($this->attachment_parser)) return $this->attachment_parser->title;
	}
	
	public function received_from(){
		if(is_object($this->attachment_parser)) return $this->attachment_parser->organization;
	}
	
	public function patient_id(){
		if(is_object($this->attachment_parser)) return $this->attachment_parser->patient_icn;
	}		
		
	public function attachment_content(){
		if(!isset($this->_attachment_content) && !$this->property_is_empty('hash') && $this->has_message()){
			foreach($this->message->attachment_files as $filename => $binary_string){
				if(sha1($binary_string) == $this->hash){
					$this->_attachment_content = $binary_string;
				}
			}
		}
		return $this->_attachment_content;
	}
	
	public function attachment_parser(){
		if(!isset($this->_attachment_parser) && !$this->property_is_empty('attachment_content')){
			if(ccd::string_matches_schema($this->attachment_content)){
				$this->_attachment_parser = new ccd($this->attachment_content);
			}
			else{
				$this->error->warning('Unknown schema for '.$this->describe());
			}
		}
		return $this->_attachment_parser;
	}
	
	
/////////////////////
// DATA MANAGEMENT
////////////////////	
	
	protected function _run_before_create(){ 
		if(!isset($this->disclosed)) $this->disclosed = time();
		return true; 
	}	
	
	protected function _values_for_save(){
		$values_for_save = parent::_values_for_save();
		
		//if the hash has changed, set the patient data
		if(array_key_exists('hash', $values_for_save)){
			$values_for_save = array_merge($values_for_save, $this->values( array('first', 'last', 'title', 'received_from', 'patient_id')));
		}
		
		//if the message has changed, set the data from the mailbox
		if(array_key_exists(static::related_foreign_key('message'), $this->altered_values)){
			$mailbox = $this->message->mailbox;
			$values_for_save['username'] = $mailbox->name;
			$values_for_save['facility'] = $mailbox->facility('name');
		}

		return $values_for_save;
	}
	

	/**
	* Validates that the values match an attachment that requires disclosure on an existing message; if no recipient is specified, the message must be a draft.
	* @return boolean
	*/
	protected function _values_are_valid_for_create_and_update(){	
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		
		$message_foreign_key = static::related_foreign_key('message');
		
		$required_fields = array( $message_foreign_key, 'hash', 'purpose', 'ssn', 'disclosed');
		foreach($required_fields as $required_field){
			if($this->property_is_empty($required_field)) return $this->error->warning(get_class($this).'::$'.$required_field.' is a required field, and must be set before saving'); 
		}
		
		//make sure that the message exists and has an attachment that matches the hash
		$message = Message::find_one($this->$message_foreign_key);
		if(!Message::is_an_entity($message)) return $this->error->warning('There is no message with id '.$this->$message_foreign_key);
		if($this->property_is_empty('recipient') && !$message->draft) return $this->error->warning($message->describe().' is not a draft');
		if($this->property_is_empty('attachment_content')) 
			return $this->error->warning($message->describe().' does not have an attachment that matches the given hash: '.$this->error->describe($this->hash));
		if(!Attachment_disclosure::content_requires_disclosure($this->attachment_content)) 
			return $this->error->warning('The attachment for '.$message->describe().' with hash '.$this->hash.' does not require disclosure');

		if(isset($this->id)) static::db()->where(static::$primary_key.' !=', $this->id);
		if(static::exists(array($message_foreign_key => $this->$message_foreign_key, 'hash' => $this->hash, 'recipient' => $this->recipient))){
			return $this->error->warning('There is already an attachment disclosure for message#'.$this->$message_foreign_key.' with hash='.$this->hash.' and recipient='.$this->error->describe($this->recipient));		
		}

		return true;
	}
	
///////////////////////////
// STATIC METHODS
///////////////////////////	
	
	/**
	* True if the content provided requires a purpose of disclosure to be provided.
	* @param string The content of an attachment
	*/
	public static function content_requires_disclosure($binary_string){
		if(!is_string($binary_string)) return should_be('string', $binary_string);
		return validates_as('is_a_c32_xml_string', $binary_string);
	}
	
	public static function update_or_create_for_message($message, $values){
		if(!Message::is_an_entity($message)) return should_be('message entity', $message);
		
		$message_foreign_key = static::related_foreign_key('message');
		$values[$message_foreign_key] = $message->id;
		$disclosure = Attachment_disclosure::find_one(array_intersect_key($values, array_flip(array($message_foreign_key, 'hash', 'recipient'))));
		
		if(Attachment_disclosure::is_an_entity($disclosure)){
			return Attachment_disclosure::update($disclosure->id, $values);
		}
		
		return Attachment_disclosure::create($values);
	}	
		
}
